/**
 * Copyright (c) 2016 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.dialogs;

import static java.text.MessageFormat.format;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.*;

import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.epp.internal.logging.aeri.ide.ILogEvent;
import org.eclipse.epp.internal.logging.aeri.ide.IProcessorDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.internal.logging.aeri.ide.processors.Processors;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Browsers;
import org.eclipse.epp.logging.aeri.core.IProblemState;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Shell;

final class ProblemStatusDialog extends MessageDialog {
    private static final String LINE_SEPARATOR = System.lineSeparator();

    private ILogEvent event;
    private List<IProcessorDescriptor> requestedDescriptors;

    ProblemStatusDialog(ILogEvent event, List<IProcessorDescriptor> reportProcessorDescriptors, Shell parentShell) {
        super(parentShell, createTitle(event), null, "", MessageDialog.INFORMATION, new String[] { IDialogConstants.OK_LABEL }, 0); //$NON-NLS-1$
        requestedDescriptors = reportProcessorDescriptors.stream().filter(descriptor -> Processors.shouldProcess(descriptor, event))
                .collect(Collectors.toList());
        setShellStyle(SWT.MODELESS | SWT.DIALOG_TRIM | SWT.RESIZE | SWT.APPLICATION_MODAL);
        this.event = event;
    }

    private static String createTitle(ILogEvent event) {
        switch (event.getInterest().getStatus()) {
        case CONFIRMED:
            return Messages.STATUS_TITLE_CONFIRMED;
        case FAILURE:
            return Messages.STATUS_TITLE_FAILURE;
        case FIXED:
            return Messages.STATUS_TITLE_FIXED;
        case IGNORED:
            return Messages.STATUS_TITLE_IGNORED;
        case INVALID:
            return Messages.STATUS_TITLE_INVALID;
        case NEEDINFO:
            return Messages.STATUS_TITLE_NEEDINFO;
        case NEW:
            return Messages.STATUS_TITLE_NEW;
        case UNCONFIRMED:
            return Messages.STATUS_TITLE_UNCONFIRMED;
        default:
            throw new IllegalArgumentException();
        }
    }

    @Override
    protected Control createMessageArea(Composite composite) {
        Image image = getImage();
        if (image != null) {
            imageLabel = new Label(composite, SWT.NULL);
            image.setBackground(imageLabel.getBackground());
            imageLabel.setImage(image);
            GridDataFactory.fillDefaults().align(SWT.CENTER, SWT.BEGINNING).applyTo(imageLabel);
        }

        Composite messageLink = new Composite(composite, SWT.NONE);
        GridDataFactory.fillDefaults().align(SWT.FILL, SWT.BEGINNING).grab(true, false).applyTo(messageLink);
        messageLink.setLayout(new GridLayout());
        Link message = new Link(messageLink, SWT.WRAP);
        GridDataFactory.fillDefaults().align(SWT.FILL, SWT.BEGINNING).grab(true, false)
                .hint(convertHorizontalDLUsToPixels(IDialogConstants.MINIMUM_MESSAGE_AREA_WIDTH), SWT.DEFAULT).applyTo(message);

        IProblemState interest = event.getInterest();
        StringBuilder builder = new StringBuilder();
        builder.append(interest.getMessage());
        message.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                Browsers.openInExternalBrowser(e.text);
            }
        });
        if (!requestedDescriptors.isEmpty()) {
            builder.append(LINE_SEPARATOR).append(LINE_SEPARATOR);
            builder.append("Additionally requested information:"); //$NON-NLS-1$
            // empty grid cell below the image
            new Label(composite, SWT.NONE);
            final Composite sections = new Composite(composite, SWT.NONE);
            gl().spacing(0, 0).applyTo(sections);
            gd().align(SWT.FILL, SWT.BEGINNING).applyTo(sections);
            for (IProcessorDescriptor descriptor : requestedDescriptors) {
                Composite section = new Composite(sections, SWT.NONE);
                gl().numColumns(1).applyTo(section);
                gd().indent(20, 0).align(SWT.FILL, SWT.BEGINNING).applyTo(section);
                CLabel processorLabel = new CLabel(section, SWT.NONE);
                processorLabel.setImage(descriptor.getImage16());
                processorLabel.setText(descriptor.getName());
                processorLabel.setToolTipText(format("{0}:\n{1}", descriptor.getName(), descriptor.getDescription())); //$NON-NLS-1$
            }
        }
        message.setText(builder.toString());
        return composite;
    }

}
