
package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Container}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableContainer.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableContainer implements Container {
  private final String id;
  private final @Nullable List<String> names;
  private final String image;
  private final @Nullable String imageId;
  private final String command;
  private final Long created;
  private final @Nullable String state;
  private final String status;
  private final @Nullable List<Container.PortMapping> ports;
  private final @Nullable Map<String, String> labels;
  private final @Nullable Long sizeRw;
  private final @Nullable Long sizeRootFs;
  private final @Nullable NetworkSettings networkSettings;
  private final @Nullable List<ContainerMount> mounts;
  private transient final String portsAsString;

  private ImmutableContainer(
      String id,
      @Nullable List<String> names,
      String image,
      @Nullable String imageId,
      String command,
      Long created,
      @Nullable String state,
      String status,
      @Nullable List<Container.PortMapping> ports,
      @Nullable Map<String, String> labels,
      @Nullable Long sizeRw,
      @Nullable Long sizeRootFs,
      @Nullable NetworkSettings networkSettings,
      @Nullable List<ContainerMount> mounts) {
    this.id = id;
    this.names = names;
    this.image = image;
    this.imageId = imageId;
    this.command = command;
    this.created = created;
    this.state = state;
    this.status = status;
    this.ports = ports;
    this.labels = labels;
    this.sizeRw = sizeRw;
    this.sizeRootFs = sizeRootFs;
    this.networkSettings = networkSettings;
    this.mounts = mounts;
    this.portsAsString = Objects.requireNonNull(Container.super.portsAsString(), "portsAsString");
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("Id")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code names} attribute
   */
  @JsonProperty("Names")
  @Override
  public @Nullable List<String> names() {
    return names;
  }

  /**
   * @return The value of the {@code image} attribute
   */
  @JsonProperty("Image")
  @Override
  public String image() {
    return image;
  }

  /**
   * @return The value of the {@code imageId} attribute
   */
  @JsonProperty("ImageID")
  @Override
  public @Nullable String imageId() {
    return imageId;
  }

  /**
   * @return The value of the {@code command} attribute
   */
  @JsonProperty("Command")
  @Override
  public String command() {
    return command;
  }

  /**
   * @return The value of the {@code created} attribute
   */
  @JsonProperty("Created")
  @Override
  public Long created() {
    return created;
  }

  /**
   * @return The value of the {@code state} attribute
   */
  @JsonProperty("State")
  @Override
  public @Nullable String state() {
    return state;
  }

  /**
   * @return The value of the {@code status} attribute
   */
  @JsonProperty("Status")
  @Override
  public String status() {
    return status;
  }

  /**
   * @return The value of the {@code ports} attribute
   */
  @JsonProperty("Ports")
  @Override
  public @Nullable List<Container.PortMapping> ports() {
    return ports;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * @return The value of the {@code sizeRw} attribute
   */
  @JsonProperty("SizeRw")
  @Override
  public @Nullable Long sizeRw() {
    return sizeRw;
  }

  /**
   * @return The value of the {@code sizeRootFs} attribute
   */
  @JsonProperty("SizeRootFs")
  @Override
  public @Nullable Long sizeRootFs() {
    return sizeRootFs;
  }

  /**
   * @return The value of the {@code networkSettings} attribute
   */
  @JsonProperty("NetworkSettings")
  @Override
  public @Nullable NetworkSettings networkSettings() {
    return networkSettings;
  }

  /**
   * @return The value of the {@code mounts} attribute
   */
  @JsonProperty("Mounts")
  @Override
  public @Nullable List<ContainerMount> mounts() {
    return mounts;
  }

  /**
   * @return The computed-at-construction value of the {@code portsAsString} attribute
   */
  @JsonProperty(value = "portsAsString", required = true)
  @JsonIgnore
  @Override
  public String portsAsString() {
    return portsAsString;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableContainer(
        newValue,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Container#names() names}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainer withNames(@Nullable String... elements) {
    if (elements == null) {
      if (this.names == null) return this;
      return new ImmutableContainer(
          this.id,
          null,
          this.image,
          this.imageId,
          this.command,
          this.created,
          this.state,
          this.status,
          this.ports,
          this.labels,
          this.sizeRw,
          this.sizeRootFs,
          this.networkSettings,
          this.mounts);
    }
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainer(
        this.id,
        newValue,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Container#names() names}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of names elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableContainer withNames(@Nullable Iterable<String> elements) {
    if (this.names == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainer(
        this.id,
        newValue,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#image() image} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for image
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withImage(String value) {
    String newValue = Objects.requireNonNull(value, "image");
    if (this.image.equals(newValue)) return this;
    return new ImmutableContainer(
        this.id,
        this.names,
        newValue,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#imageId() imageId} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for imageId (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withImageId(@Nullable String value) {
    if (Objects.equals(this.imageId, value)) return this;
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        value,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#command() command} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for command
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withCommand(String value) {
    String newValue = Objects.requireNonNull(value, "command");
    if (this.command.equals(newValue)) return this;
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        newValue,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#created() created} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for created
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withCreated(Long value) {
    Long newValue = Objects.requireNonNull(value, "created");
    if (this.created.equals(newValue)) return this;
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        newValue,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#state() state} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for state (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withState(@Nullable String value) {
    if (Objects.equals(this.state, value)) return this;
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        value,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#status() status} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for status
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withStatus(String value) {
    String newValue = Objects.requireNonNull(value, "status");
    if (this.status.equals(newValue)) return this;
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        newValue,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Container#ports() ports}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainer withPorts(@Nullable Container.PortMapping... elements) {
    if (elements == null) {
      if (this.ports == null) return this;
      return new ImmutableContainer(
          this.id,
          this.names,
          this.image,
          this.imageId,
          this.command,
          this.created,
          this.state,
          this.status,
          null,
          this.labels,
          this.sizeRw,
          this.sizeRootFs,
          this.networkSettings,
          this.mounts);
    }
    List<Container.PortMapping> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        newValue,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Container#ports() ports}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of ports elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableContainer withPorts(@Nullable Iterable<? extends Container.PortMapping> elements) {
    if (this.ports == elements) return this;
    @Nullable List<Container.PortMapping> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        newValue,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by replacing the {@link Container#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to replace the labels map
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableContainer withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        newValue,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#sizeRw() sizeRw} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for sizeRw (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withSizeRw(@Nullable Long value) {
    if (Objects.equals(this.sizeRw, value)) return this;
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        value,
        this.sizeRootFs,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#sizeRootFs() sizeRootFs} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for sizeRootFs (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withSizeRootFs(@Nullable Long value) {
    if (Objects.equals(this.sizeRootFs, value)) return this;
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        value,
        this.networkSettings,
        this.mounts);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Container#networkSettings() networkSettings} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for networkSettings (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainer withNetworkSettings(@Nullable NetworkSettings value) {
    if (this.networkSettings == value) return this;
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        value,
        this.mounts);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Container#mounts() mounts}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainer withMounts(@Nullable ContainerMount... elements) {
    if (elements == null) {
      if (this.mounts == null) return this;
      return new ImmutableContainer(
          this.id,
          this.names,
          this.image,
          this.imageId,
          this.command,
          this.created,
          this.state,
          this.status,
          this.ports,
          this.labels,
          this.sizeRw,
          this.sizeRootFs,
          this.networkSettings,
          null);
    }
    List<ContainerMount> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Container#mounts() mounts}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of mounts elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableContainer withMounts(@Nullable Iterable<? extends ContainerMount> elements) {
    if (this.mounts == elements) return this;
    @Nullable List<ContainerMount> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainer(
        this.id,
        this.names,
        this.image,
        this.imageId,
        this.command,
        this.created,
        this.state,
        this.status,
        this.ports,
        this.labels,
        this.sizeRw,
        this.sizeRootFs,
        this.networkSettings,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableContainer} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableContainer
        && equalsByValue((ImmutableContainer) another);
  }

  private boolean equalsByValue(ImmutableContainer another) {
    return id.equals(another.id)
        && Objects.equals(names, another.names)
        && image.equals(another.image)
        && Objects.equals(imageId, another.imageId)
        && command.equals(another.command)
        && created.equals(another.created)
        && Objects.equals(state, another.state)
        && status.equals(another.status)
        && Objects.equals(ports, another.ports)
        && Objects.equals(labels, another.labels)
        && Objects.equals(sizeRw, another.sizeRw)
        && Objects.equals(sizeRootFs, another.sizeRootFs)
        && Objects.equals(networkSettings, another.networkSettings)
        && Objects.equals(mounts, another.mounts)
        && portsAsString.equals(another.portsAsString);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code names}, {@code image}, {@code imageId}, {@code command}, {@code created}, {@code state}, {@code status}, {@code ports}, {@code labels}, {@code sizeRw}, {@code sizeRootFs}, {@code networkSettings}, {@code mounts}, {@code portsAsString}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + id.hashCode();
    h += (h << 5) + Objects.hashCode(names);
    h += (h << 5) + image.hashCode();
    h += (h << 5) + Objects.hashCode(imageId);
    h += (h << 5) + command.hashCode();
    h += (h << 5) + created.hashCode();
    h += (h << 5) + Objects.hashCode(state);
    h += (h << 5) + status.hashCode();
    h += (h << 5) + Objects.hashCode(ports);
    h += (h << 5) + Objects.hashCode(labels);
    h += (h << 5) + Objects.hashCode(sizeRw);
    h += (h << 5) + Objects.hashCode(sizeRootFs);
    h += (h << 5) + Objects.hashCode(networkSettings);
    h += (h << 5) + Objects.hashCode(mounts);
    h += (h << 5) + portsAsString.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Container} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Container{"
        + "id=" + id
        + ", names=" + names
        + ", image=" + image
        + ", imageId=" + imageId
        + ", command=" + command
        + ", created=" + created
        + ", state=" + state
        + ", status=" + status
        + ", ports=" + ports
        + ", labels=" + labels
        + ", sizeRw=" + sizeRw
        + ", sizeRootFs=" + sizeRootFs
        + ", networkSettings=" + networkSettings
        + ", mounts=" + mounts
        + ", portsAsString=" + portsAsString
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Container} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Container instance
   */
  public static ImmutableContainer copyOf(Container instance) {
    if (instance instanceof ImmutableContainer) {
      return (ImmutableContainer) instance;
    }
    return ImmutableContainer.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableContainer ImmutableContainer}.
   * <pre>
   * ImmutableContainer.builder()
   *    .id(String) // required {@link Container#id() id}
   *    .names(List&lt;String&gt; | null) // nullable {@link Container#names() names}
   *    .image(String) // required {@link Container#image() image}
   *    .imageId(String | null) // nullable {@link Container#imageId() imageId}
   *    .command(String) // required {@link Container#command() command}
   *    .created(Long) // required {@link Container#created() created}
   *    .state(String | null) // nullable {@link Container#state() state}
   *    .status(String) // required {@link Container#status() status}
   *    .ports(List&lt;org.mandas.docker.client.messages.Container.PortMapping&gt; | null) // nullable {@link Container#ports() ports}
   *    .labels(Map&lt;String, String&gt; | null) // nullable {@link Container#labels() labels}
   *    .sizeRw(Long | null) // nullable {@link Container#sizeRw() sizeRw}
   *    .sizeRootFs(Long | null) // nullable {@link Container#sizeRootFs() sizeRootFs}
   *    .networkSettings(org.mandas.docker.client.messages.NetworkSettings | null) // nullable {@link Container#networkSettings() networkSettings}
   *    .mounts(List&lt;org.mandas.docker.client.messages.ContainerMount&gt; | null) // nullable {@link Container#mounts() mounts}
   *    .build();
   * </pre>
   * @return A new ImmutableContainer builder
   */
  public static ImmutableContainer.Builder builder() {
    return new ImmutableContainer.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableContainer ImmutableContainer}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ID = 0x1L;
    private static final long INIT_BIT_IMAGE = 0x2L;
    private static final long INIT_BIT_COMMAND = 0x4L;
    private static final long INIT_BIT_CREATED = 0x8L;
    private static final long INIT_BIT_STATUS = 0x10L;
    private long initBits = 0x1fL;

    private String id;
    private List<String> names = null;
    private String image;
    private String imageId;
    private String command;
    private Long created;
    private String state;
    private String status;
    private List<Container.PortMapping> ports = null;
    private Map<String, String> labels = null;
    private Long sizeRw;
    private Long sizeRootFs;
    private NetworkSettings networkSettings;
    private List<ContainerMount> mounts = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Container} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Container instance) {
      Objects.requireNonNull(instance, "instance");
      this.id(instance.id());
      @Nullable List<String> namesValue = instance.names();
      if (namesValue != null) {
        addAllNames(namesValue);
      }
      this.image(instance.image());
      @Nullable String imageIdValue = instance.imageId();
      if (imageIdValue != null) {
        imageId(imageIdValue);
      }
      this.command(instance.command());
      this.created(instance.created());
      @Nullable String stateValue = instance.state();
      if (stateValue != null) {
        state(stateValue);
      }
      this.status(instance.status());
      @Nullable List<Container.PortMapping> portsValue = instance.ports();
      if (portsValue != null) {
        addAllPorts(portsValue);
      }
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      @Nullable Long sizeRwValue = instance.sizeRw();
      if (sizeRwValue != null) {
        sizeRw(sizeRwValue);
      }
      @Nullable Long sizeRootFsValue = instance.sizeRootFs();
      if (sizeRootFsValue != null) {
        sizeRootFs(sizeRootFsValue);
      }
      @Nullable NetworkSettings networkSettingsValue = instance.networkSettings();
      if (networkSettingsValue != null) {
        networkSettings(networkSettingsValue);
      }
      @Nullable List<ContainerMount> mountsValue = instance.mounts();
      if (mountsValue != null) {
        addAllMounts(mountsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Container#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Id")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Adds one element to {@link Container#names() names} list.
     * @param element A names element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder name(String element) {
      if (this.names == null) {
        this.names = new ArrayList<String>();
      }
      this.names.add(Objects.requireNonNull(element, "names element"));
      return this;
    }

    /**
     * Adds elements to {@link Container#names() names} list.
     * @param elements An array of names elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder names(String... elements) {
      if (this.names == null) {
        this.names = new ArrayList<String>();
      }
      for (String element : elements) {
        this.names.add(Objects.requireNonNull(element, "names element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Container#names() names} list.
     * @param elements An iterable of names elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Names")
    public final Builder names(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.names = null;
        return this;
      }
      this.names = new ArrayList<String>();
      return addAllNames(elements);
    }

    /**
     * Adds elements to {@link Container#names() names} list.
     * @param elements An iterable of names elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllNames(Iterable<String> elements) {
      Objects.requireNonNull(elements, "names element");
      if (this.names == null) {
        this.names = new ArrayList<String>();
      }
      for (String element : elements) {
        this.names.add(Objects.requireNonNull(element, "names element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Container#image() image} attribute.
     * @param image The value for image 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Image")
    public final Builder image(String image) {
      this.image = Objects.requireNonNull(image, "image");
      initBits &= ~INIT_BIT_IMAGE;
      return this;
    }

    /**
     * Initializes the value for the {@link Container#imageId() imageId} attribute.
     * @param imageId The value for imageId (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ImageID")
    public final Builder imageId(@Nullable String imageId) {
      this.imageId = imageId;
      return this;
    }

    /**
     * Initializes the value for the {@link Container#command() command} attribute.
     * @param command The value for command 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Command")
    public final Builder command(String command) {
      this.command = Objects.requireNonNull(command, "command");
      initBits &= ~INIT_BIT_COMMAND;
      return this;
    }

    /**
     * Initializes the value for the {@link Container#created() created} attribute.
     * @param created The value for created 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Created")
    public final Builder created(Long created) {
      this.created = Objects.requireNonNull(created, "created");
      initBits &= ~INIT_BIT_CREATED;
      return this;
    }

    /**
     * Initializes the value for the {@link Container#state() state} attribute.
     * @param state The value for state (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("State")
    public final Builder state(@Nullable String state) {
      this.state = state;
      return this;
    }

    /**
     * Initializes the value for the {@link Container#status() status} attribute.
     * @param status The value for status 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Status")
    public final Builder status(String status) {
      this.status = Objects.requireNonNull(status, "status");
      initBits &= ~INIT_BIT_STATUS;
      return this;
    }

    /**
     * Adds one element to {@link Container#ports() ports} list.
     * @param element A ports element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder port(Container.PortMapping element) {
      if (this.ports == null) {
        this.ports = new ArrayList<Container.PortMapping>();
      }
      this.ports.add(Objects.requireNonNull(element, "ports element"));
      return this;
    }

    /**
     * Adds elements to {@link Container#ports() ports} list.
     * @param elements An array of ports elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder ports(Container.PortMapping... elements) {
      if (this.ports == null) {
        this.ports = new ArrayList<Container.PortMapping>();
      }
      for (Container.PortMapping element : elements) {
        this.ports.add(Objects.requireNonNull(element, "ports element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Container#ports() ports} list.
     * @param elements An iterable of ports elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Ports")
    public final Builder ports(@Nullable Iterable<? extends Container.PortMapping> elements) {
      if (elements == null) {
        this.ports = null;
        return this;
      }
      this.ports = new ArrayList<Container.PortMapping>();
      return addAllPorts(elements);
    }

    /**
     * Adds elements to {@link Container#ports() ports} list.
     * @param elements An iterable of ports elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllPorts(Iterable<? extends Container.PortMapping> elements) {
      Objects.requireNonNull(elements, "ports element");
      if (this.ports == null) {
        this.ports = new ArrayList<Container.PortMapping>();
      }
      for (Container.PortMapping element : elements) {
        this.ports.add(Objects.requireNonNull(element, "ports element"));
      }
      return this;
    }

    /**
     * Put one entry to the {@link Container#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, value == null ? "labels value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link Container#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link Container#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link Container#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Container#sizeRw() sizeRw} attribute.
     * @param sizeRw The value for sizeRw (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("SizeRw")
    public final Builder sizeRw(@Nullable Long sizeRw) {
      this.sizeRw = sizeRw;
      return this;
    }

    /**
     * Initializes the value for the {@link Container#sizeRootFs() sizeRootFs} attribute.
     * @param sizeRootFs The value for sizeRootFs (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("SizeRootFs")
    public final Builder sizeRootFs(@Nullable Long sizeRootFs) {
      this.sizeRootFs = sizeRootFs;
      return this;
    }

    /**
     * Initializes the value for the {@link Container#networkSettings() networkSettings} attribute.
     * @param networkSettings The value for networkSettings (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("NetworkSettings")
    public final Builder networkSettings(@Nullable NetworkSettings networkSettings) {
      this.networkSettings = networkSettings;
      return this;
    }

    /**
     * Adds one element to {@link Container#mounts() mounts} list.
     * @param element A mounts element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder mount(ContainerMount element) {
      if (this.mounts == null) {
        this.mounts = new ArrayList<ContainerMount>();
      }
      this.mounts.add(Objects.requireNonNull(element, "mounts element"));
      return this;
    }

    /**
     * Adds elements to {@link Container#mounts() mounts} list.
     * @param elements An array of mounts elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder mounts(ContainerMount... elements) {
      if (this.mounts == null) {
        this.mounts = new ArrayList<ContainerMount>();
      }
      for (ContainerMount element : elements) {
        this.mounts.add(Objects.requireNonNull(element, "mounts element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Container#mounts() mounts} list.
     * @param elements An iterable of mounts elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Mounts")
    public final Builder mounts(@Nullable Iterable<? extends ContainerMount> elements) {
      if (elements == null) {
        this.mounts = null;
        return this;
      }
      this.mounts = new ArrayList<ContainerMount>();
      return addAllMounts(elements);
    }

    /**
     * Adds elements to {@link Container#mounts() mounts} list.
     * @param elements An iterable of mounts elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllMounts(Iterable<? extends ContainerMount> elements) {
      Objects.requireNonNull(elements, "mounts element");
      if (this.mounts == null) {
        this.mounts = new ArrayList<ContainerMount>();
      }
      for (ContainerMount element : elements) {
        this.mounts.add(Objects.requireNonNull(element, "mounts element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableContainer ImmutableContainer}.
     * @return An immutable instance of Container
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableContainer build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableContainer(
          id,
          names == null ? null : createUnmodifiableList(true, names),
          image,
          imageId,
          command,
          created,
          state,
          status,
          ports == null ? null : createUnmodifiableList(true, ports),
          labels == null ? null : createUnmodifiableMap(false, false, labels),
          sizeRw,
          sizeRootFs,
          networkSettings,
          mounts == null ? null : createUnmodifiableList(true, mounts));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      if ((initBits & INIT_BIT_IMAGE) != 0) attributes.add("image");
      if ((initBits & INIT_BIT_COMMAND) != 0) attributes.add("command");
      if ((initBits & INIT_BIT_CREATED) != 0) attributes.add("created");
      if ((initBits & INIT_BIT_STATUS) != 0) attributes.add("status");
      return "Cannot build Container, some of required attributes are not set " + attributes;
    }
  }

  /**
   * Immutable implementation of {@link Container.PortMapping}.
   * <p>
   * Use the builder to create immutable instances:
   * {@code ImmutableContainer.PortMapping.builder()}.
   */
  static final class PortMapping implements Container.PortMapping {
    private final int privatePort;
    private final int publicPort;
    private final String type;
    private final @Nullable String ip;

    private PortMapping(ImmutableContainer.PortMapping.Builder builder) {
      this.type = builder.type;
      this.ip = builder.ip;
      if (builder.privatePortIsSet()) {
        initShim.privatePort(builder.privatePort);
      }
      if (builder.publicPortIsSet()) {
        initShim.publicPort(builder.publicPort);
      }
      this.privatePort = initShim.privatePort();
      this.publicPort = initShim.publicPort();
      this.initShim = null;
    }

    private PortMapping(
        int privatePort,
        int publicPort,
        String type,
        @Nullable String ip) {
      this.privatePort = privatePort;
      this.publicPort = publicPort;
      this.type = type;
      this.ip = ip;
      this.initShim = null;
    }

    private static final byte STAGE_INITIALIZING = -1;
    private static final byte STAGE_UNINITIALIZED = 0;
    private static final byte STAGE_INITIALIZED = 1;
    private transient volatile InitShim initShim = new InitShim();

    private final class InitShim {
      private byte privatePortBuildStage = STAGE_UNINITIALIZED;
      private int privatePort;

      int privatePort() {
        if (privatePortBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
        if (privatePortBuildStage == STAGE_UNINITIALIZED) {
          privatePortBuildStage = STAGE_INITIALIZING;
          this.privatePort = privatePortInitialize();
          privatePortBuildStage = STAGE_INITIALIZED;
        }
        return this.privatePort;
      }

      void privatePort(int privatePort) {
        this.privatePort = privatePort;
        privatePortBuildStage = STAGE_INITIALIZED;
      }

      private byte publicPortBuildStage = STAGE_UNINITIALIZED;
      private int publicPort;

      int publicPort() {
        if (publicPortBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
        if (publicPortBuildStage == STAGE_UNINITIALIZED) {
          publicPortBuildStage = STAGE_INITIALIZING;
          this.publicPort = publicPortInitialize();
          publicPortBuildStage = STAGE_INITIALIZED;
        }
        return this.publicPort;
      }

      void publicPort(int publicPort) {
        this.publicPort = publicPort;
        publicPortBuildStage = STAGE_INITIALIZED;
      }

      private String formatInitCycleMessage() {
        List<String> attributes = new ArrayList<>();
        if (privatePortBuildStage == STAGE_INITIALIZING) attributes.add("privatePort");
        if (publicPortBuildStage == STAGE_INITIALIZING) attributes.add("publicPort");
        return "Cannot build PortMapping, attribute initializers form cycle " + attributes;
      }
    }

    private int privatePortInitialize() {
      return Container.PortMapping.super.privatePort();
    }

    private int publicPortInitialize() {
      return Container.PortMapping.super.publicPort();
    }

    /**
     * @return The value of the {@code privatePort} attribute
     */
    @JsonProperty("PrivatePort")
    @Override
    public int privatePort() {
      InitShim shim = this.initShim;
      return shim != null
          ? shim.privatePort()
          : this.privatePort;
    }

    /**
     * @return The value of the {@code publicPort} attribute
     */
    @JsonProperty("PublicPort")
    @Override
    public int publicPort() {
      InitShim shim = this.initShim;
      return shim != null
          ? shim.publicPort()
          : this.publicPort;
    }

    /**
     * @return The value of the {@code type} attribute
     */
    @JsonProperty("Type")
    @Override
    public String type() {
      return type;
    }

    /**
     * @return The value of the {@code ip} attribute
     */
    @JsonProperty("IP")
    @Override
    public @Nullable String ip() {
      return ip;
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Container.PortMapping#privatePort() privatePort} attribute.
     * A value equality check is used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for privatePort
     * @return A modified copy or the {@code this} object
     */
    public final ImmutableContainer.PortMapping withPrivatePort(int value) {
      if (this.privatePort == value) return this;
      return new ImmutableContainer.PortMapping(value, this.publicPort, this.type, this.ip);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Container.PortMapping#publicPort() publicPort} attribute.
     * A value equality check is used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for publicPort
     * @return A modified copy or the {@code this} object
     */
    public final ImmutableContainer.PortMapping withPublicPort(int value) {
      if (this.publicPort == value) return this;
      return new ImmutableContainer.PortMapping(this.privatePort, value, this.type, this.ip);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Container.PortMapping#type() type} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for type
     * @return A modified copy or the {@code this} object
     */
    public final ImmutableContainer.PortMapping withType(String value) {
      String newValue = Objects.requireNonNull(value, "type");
      if (this.type.equals(newValue)) return this;
      return new ImmutableContainer.PortMapping(this.privatePort, this.publicPort, newValue, this.ip);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Container.PortMapping#ip() ip} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for ip (can be {@code null})
     * @return A modified copy or the {@code this} object
     */
    public final ImmutableContainer.PortMapping withIp(@Nullable String value) {
      if (Objects.equals(this.ip, value)) return this;
      return new ImmutableContainer.PortMapping(this.privatePort, this.publicPort, this.type, value);
    }

    /**
     * This instance is equal to all instances of {@code PortMapping} that have equal attribute values.
     * @return {@code true} if {@code this} is equal to {@code another} instance
     */
    @Override
    public boolean equals(Object another) {
      if (this == another) return true;
      return another instanceof ImmutableContainer.PortMapping
          && equalsByValue((ImmutableContainer.PortMapping) another);
    }

    private boolean equalsByValue(ImmutableContainer.PortMapping another) {
      return privatePort == another.privatePort
          && publicPort == another.publicPort
          && type.equals(another.type)
          && Objects.equals(ip, another.ip);
    }

    /**
     * Computes a hash code from attributes: {@code privatePort}, {@code publicPort}, {@code type}, {@code ip}.
     * @return hashCode value
     */
    @Override
    public int hashCode() {
      int h = 5381;
      h += (h << 5) + privatePort;
      h += (h << 5) + publicPort;
      h += (h << 5) + type.hashCode();
      h += (h << 5) + Objects.hashCode(ip);
      return h;
    }

    /**
     * Prints the immutable value {@code PortMapping} with attribute values.
     * @return A string representation of the value
     */
    @Override
    public String toString() {
      return "PortMapping{"
          + "privatePort=" + privatePort
          + ", publicPort=" + publicPort
          + ", type=" + type
          + ", ip=" + ip
          + "}";
    }

    /**
     * Creates an immutable copy of a {@link Container.PortMapping} value.
     * Uses accessors to get values to initialize the new immutable instance.
     * If an instance is already immutable, it is returned as is.
     * @param instance The instance to copy
     * @return A copied immutable PortMapping instance
     */
    public static ImmutableContainer.PortMapping copyOf(Container.PortMapping instance) {
      if (instance instanceof ImmutableContainer.PortMapping) {
        return (ImmutableContainer.PortMapping) instance;
      }
      return ImmutableContainer.PortMapping.builder()
          .from(instance)
          .build();
    }

    /**
     * Creates a builder for {@link ImmutableContainer.PortMapping PortMapping}.
     * <pre>
     * ImmutableContainer.PortMapping.builder()
     *    .privatePort(int) // optional {@link Container.PortMapping#privatePort() privatePort}
     *    .publicPort(int) // optional {@link Container.PortMapping#publicPort() publicPort}
     *    .type(String) // required {@link Container.PortMapping#type() type}
     *    .ip(String | null) // nullable {@link Container.PortMapping#ip() ip}
     *    .build();
     * </pre>
     * @return A new PortMapping builder
     */
    public static ImmutableContainer.PortMapping.Builder builder() {
      return new ImmutableContainer.PortMapping.Builder();
    }

    /**
     * Builds instances of type {@link ImmutableContainer.PortMapping PortMapping}.
     * Initialize attributes and then invoke the {@link #build()} method to create an
     * immutable instance.
     * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
     * but instead used immediately to create instances.</em>
     */
    static final class Builder {
      private static final long INIT_BIT_TYPE = 0x1L;
      private static final long OPT_BIT_PRIVATE_PORT = 0x1L;
      private static final long OPT_BIT_PUBLIC_PORT = 0x2L;
      private long initBits = 0x1L;
      private long optBits;

      private int privatePort;
      private int publicPort;
      private String type;
      private String ip;

      private Builder() {
      }

      /**
       * Fill a builder with attribute values from the provided {@code PortMapping} instance.
       * Regular attribute values will be replaced with those from the given instance.
       * Absent optional values will not replace present values.
       * @param instance The instance from which to copy values
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder from(Container.PortMapping instance) {
        Objects.requireNonNull(instance, "instance");
        this.privatePort(instance.privatePort());
        this.publicPort(instance.publicPort());
        this.type(instance.type());
        @Nullable String ipValue = instance.ip();
        if (ipValue != null) {
          ip(ipValue);
        }
        return this;
      }

      /**
       * Initializes the value for the {@link Container.PortMapping#privatePort() privatePort} attribute.
       * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Container.PortMapping#privatePort() privatePort}.</em>
       * @param privatePort The value for privatePort 
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("PrivatePort")
      public final Builder privatePort(int privatePort) {
        this.privatePort = privatePort;
        optBits |= OPT_BIT_PRIVATE_PORT;
        return this;
      }

      /**
       * Initializes the value for the {@link Container.PortMapping#publicPort() publicPort} attribute.
       * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Container.PortMapping#publicPort() publicPort}.</em>
       * @param publicPort The value for publicPort 
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("PublicPort")
      public final Builder publicPort(int publicPort) {
        this.publicPort = publicPort;
        optBits |= OPT_BIT_PUBLIC_PORT;
        return this;
      }

      /**
       * Initializes the value for the {@link Container.PortMapping#type() type} attribute.
       * @param type The value for type 
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("Type")
      public final Builder type(String type) {
        this.type = Objects.requireNonNull(type, "type");
        initBits &= ~INIT_BIT_TYPE;
        return this;
      }

      /**
       * Initializes the value for the {@link Container.PortMapping#ip() ip} attribute.
       * @param ip The value for ip (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("IP")
      public final Builder ip(@Nullable String ip) {
        this.ip = ip;
        return this;
      }

      /**
       * Builds a new {@link ImmutableContainer.PortMapping PortMapping}.
       * @return An immutable instance of PortMapping
       * @throws java.lang.IllegalStateException if any required attributes are missing
       */
      public ImmutableContainer.PortMapping build() {
        if (initBits != 0) {
          throw new IllegalStateException(formatRequiredAttributesMessage());
        }
        return new ImmutableContainer.PortMapping(this);
      }

      private boolean privatePortIsSet() {
        return (optBits & OPT_BIT_PRIVATE_PORT) != 0;
      }

      private boolean publicPortIsSet() {
        return (optBits & OPT_BIT_PUBLIC_PORT) != 0;
      }

      private String formatRequiredAttributesMessage() {
        List<String> attributes = new ArrayList<>();
        if ((initBits & INIT_BIT_TYPE) != 0) attributes.add("type");
        return "Cannot build PortMapping, some of required attributes are not set " + attributes;
      }
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
